# -*- coding: utf-8 -*-

# Author: Roberto Alsina <roberto.alsina@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the start function."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import gui
from ubuntuone.controlpanel.tests import TestCase


class FakeThing(object):

    """A fake thing."""

    shown = False
    size = lambda *a: gui.QtCore.QSize(123456, 654321)
    style = None

    def __init__(self):
        self.args = []
        self.updates_checked = defer.Deferred()

    def __call__(self, *args, **kwargs):
        self.args.append((args, kwargs))
        return self

    def show(self):
        """Show."""
        self.shown = True

    # Invalid name "setGeometry"
    # pylint: disable=C0103

    def setGeometry(self, style):
        """Save the new geometry."""
        self.style = style

    def check_updates(self):
        """Fake the check updates call."""
        self.updates_checked.callback(None)
        return self.updates_checked


class StartTestCase(TestCase):
    """Test the qt control panel."""

    timeout = 2

    @defer.inlineCallbacks
    def setUp(self):
        yield super(StartTestCase, self).setUp()
        self.main_window = FakeThing()
        self.tray_icon = FakeThing()
        self.patch(gui, "MainWindow", self.main_window)
        self.patch(gui, "TrayIcon", self.tray_icon)

    def close_cb(self):
        """A dummy close callback."""

    def test_minimized(self):
        """Test behaviour with minimized=True."""
        gui.start(close_callback=self.close_cb,
                  minimized=True, with_icon=True)
        kwargs = {'close_callback': self.close_cb, 'window': None}
        self.assertEqual(self.tray_icon.args, [((), kwargs)])
        self.assertEqual(self.main_window.args, [])

    def test_with_icon(self):
        """Test behaviour with with_icon=True."""
        gui.start(close_callback=self.close_cb,
                  with_icon=True, minimized=False)
        kwargs = {'close_callback': self.close_cb, 'window': self.main_window}
        self.assertEqual(self.main_window.args, [((), {'installer': False})])
        self.assertEqual(self.tray_icon.args, [((), kwargs)])

    def test_both_false(self):
        """Test behaviour when with_icon and minimized are False."""
        gui.start(close_callback=self.close_cb,
                  with_icon=False, minimized=False)
        # Should be called
        self.assertNotEqual(self.main_window.args, [])
        # Should not be called
        self.assertEqual(self.tray_icon.args, [])

    def test_both_true(self):
        """Test behaviour when with_icon and minimized are True."""
        gui.start(close_callback=self.close_cb,
                  with_icon=True, minimized=True)
        kwargs = {'close_callback': self.close_cb, 'window': None}
        self.assertEqual(self.tray_icon.args, [((), kwargs)])
        self.assertEqual(self.main_window.args, [])

    def test_center_window(self):
        """The main window should be centered."""
        gui.start(close_callback=self.close_cb)
        app = gui.QtGui.QApplication.instance()
        expected = gui.QtGui.QStyle.alignedRect(gui.QtCore.Qt.LeftToRight,
                        gui.QtCore.Qt.AlignCenter, self.main_window.size(),
                        app.desktop().availableGeometry())
        self.assertEqual(self.main_window.style, expected)

    @defer.inlineCallbacks
    def test_updates_are_checked(self):
        """When creating the window, updates are checked."""
        gui.start(close_callback=self.close_cb)

        # a timeout in this test means that the check_updates method
        # was not called
        yield self.main_window.updates_checked
