# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

from __future__ import division

from PyQt4 import QtCore
from twisted.internet import defer

from ubuntuone.controlpanel.backend import AUTOCONNECT_KEY
from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import (
    FACEBOOK_LINK,
    GET_HELP_ONLINE,
    GET_MORE_STORAGE,
    GET_STORAGE_LINK,
    GET_SUPPORT_LINK,
    GREETING,
    humanize,
    MAIN_ACCOUNT_TAB,
    MAIN_DEVICES_TAB,
    MAIN_FOLDERS_TAB,
    MAIN_PREFERENCES_TAB,
    PERCENTAGE_LABEL,
    show_quota_warning,
    TALK_TO_US,
    TWITTER_LINK,
    USAGE_LABEL,
)
from ubuntuone.controlpanel.gui import qt
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin
from ubuntuone.controlpanel.gui.qt.ui import controlpanel_ui


logger = setup_logging('qt.controlpanel')

NAME_STYLE = u'<br><span style=" font-size:xx-large;">%s!</span>'
PERCENTAGE_STYLE = u'<span style=" font-size:x-large;">%.0f%%</span>'


class ControlPanel(UbuntuOneBin):
    """The Control Panel widget"""

    ui_class = controlpanel_ui
    logger = logger

    finished = QtCore.pyqtSignal()

    def _setup(self):
        """Do some extra setupping for the UI."""
        self.ui.get_more_space_button.setText(GET_MORE_STORAGE)
        self.ui.get_more_space_button.uri = GET_STORAGE_LINK

        self.ui.help_button.setText(GET_HELP_ONLINE)
        self.ui.help_button.uri = GET_SUPPORT_LINK

        self.ui.follow_us_label.setText(TALK_TO_US)

        self.ui.devices_tab.localDeviceRemoved.connect(
            self.on_credentials_not_found)

        self.ui.tab_widget.setTabText(
            self.ui.tab_widget.indexOf(self.ui.folders_tab), MAIN_FOLDERS_TAB)
        self.ui.tab_widget.setTabText(
            self.ui.tab_widget.indexOf(self.ui.devices_tab), MAIN_DEVICES_TAB)
        self.ui.tab_widget.setTabText(
            self.ui.tab_widget.indexOf(self.ui.preferences_tab),
            MAIN_PREFERENCES_TAB)
        self.ui.tab_widget.setTabText(
            self.ui.tab_widget.indexOf(self.ui.account_tab), MAIN_ACCOUNT_TAB)

    @defer.inlineCallbacks
    def connect_file_sync(self):
        """Connect file sync service if the setting autoconnect is enabled."""
        settings = yield self.backend.file_sync_settings_info()
        if settings.get(AUTOCONNECT_KEY, False):
            yield self.backend.connect_files()

    @log_call(logger.debug)
    def on_credentials_not_found(self):
        """Credentials are not found or were removed."""
        self.ui.wizard.restart()
        self.ui.switcher.setCurrentWidget(self.ui.wizard)
        self.is_processing = False

    @defer.inlineCallbacks
    @log_call(logger.debug)
    def on_credentials_found(self):
        """Credentials are not found or were removed."""
        folders_tab_idx = self.ui.tab_widget.indexOf(self.ui.folders_tab)
        self.ui.tab_widget.setCurrentIndex(folders_tab_idx)
        self.ui.switcher.setCurrentWidget(self.ui.management)
        self.connect_file_sync()
        self.is_processing = False

        info = yield self.backend.account_info()
        self.process_info(info)

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load info."""
        self.is_processing = True
        credentials = yield self.backend.get_credentials()
        if not credentials:
            self.on_credentials_not_found()
        else:
            self.on_credentials_found()

    @log_call(logger.debug)
    def process_info(self, info):
        """Process and display the account info."""
        name = NAME_STYLE % info['name']
        self.ui.greeting_label.setText(GREETING % {'user_display_name': name})

        used = int(info['quota_used'])
        total = int(info['quota_total'])
        percentage_value = ((used / total) * 100)
        percentage = {'percentage': PERCENTAGE_STYLE % percentage_value}
        show_warning = show_quota_warning(used, total)
        data = {'used': humanize(used), 'total': humanize(total)}
        self.ui.percentage_usage_label.setText(PERCENTAGE_LABEL % percentage)
        self.ui.quota_usage_label.setText(USAGE_LABEL % data)
        self._update_quota(show_warning)

    @log_call(logger.debug)
    def _update_quota(self, show_warning=False):
        """Update the quota info."""
        logger.debug('ManagementPanel: show warning in quota %r.',
            show_warning)
        self.ui.percentage_usage_label.setProperty("OverQuota", show_warning)
        self.ui.quota_usage_label.setProperty("OverQuota", show_warning)
        self.ui.percentage_usage_label.style().unpolish(
            self.ui.percentage_usage_label)
        self.ui.percentage_usage_label.style().polish(
            self.ui.percentage_usage_label)
        self.ui.quota_usage_label.style().unpolish(
            self.ui.quota_usage_label)
        self.ui.quota_usage_label.style().polish(self.ui.quota_usage_label)

    @QtCore.pyqtSlot()
    def on_twitter_button_clicked(self):
        """The twitter button was clicked."""
        qt.uri_hook(TWITTER_LINK)

    @QtCore.pyqtSlot()
    def on_facebook_button_clicked(self):
        """The facebook button was clicked."""
        qt.uri_hook(FACEBOOK_LINK)

    @log_call(logger.warning)
    def on_wizard_rejected(self):
        """Let clients know that we're done."""
        self.finished.emit()

    @log_call(logger.info)
    def on_wizard_finished(self, status):
        """Move to controlpanel if wizard ended successfully."""
        self.on_credentials_found()

    @log_call(logger.info)
    def start_from_license(self):
        """Use the license page as first page."""
        license_id = self.ui.wizard.pages[self.ui.wizard.license_page]
        self.ui.wizard.setStartId(license_id)
        self.ui.wizard.restart()
