/* 	tweakrelocsdynamic.c: 
 *		support code to relocate symbols dynamically to needed place.
 *		Mainly for symbols in binary files which location found run-time
 *
 *	Author: A. Chentsov
 * 	Copying: LGPL
 *
 *	Exports: 
 *		initialize_map (char *prog), 
 *		relocate_symbol (char *name, void *newval);
 *
 * 	ToDo:
 *		add support for RELA sections
 */

#include <sys/types.h>
#include <stdio.h>
#include <elf.h>

#include <sys/mman.h>

#include <string.h>

#include <unistd.h>
#include <fcntl.h>

#define PREFIX "dynreloc"
#include "log.h"

static Elf32_Ehdr *image_header;
static Elf32_Shdr *section_headers;
static Elf32_Shdr *symbols_hdr;
static Elf32_Sym *syms;
static char *strtab;
static char *sec_strings;

#ifdef DYNAMICRELOC_DEBUG
#  define DEBUG_DRELOCS(args...) 	LOGDEBUG (2,args)
#else
#  define DEBUG_DRELOCS(args...)		do {} while (0)
#endif

struct symbol {
	char *name;
	int idx;
};	

static void *mapimage (char *filename) 
{
	int fd = open (filename, O_RDONLY);
	if (fd < 0) {
		LOGERR ("Bad file name %s\n", filename);
		return MAP_FAILED;
	}

	void *file = mmap (
		NULL,
		lseek (fd, 0, SEEK_END),
		PROT_READ,
		MAP_SHARED,
		fd,
		0
	);

	if (file == MAP_FAILED) {
		LOGSYSERR (Error, "mmap");
	}

	close (fd);

	return file;
}


/* image initialiazation
 * takes program name, returns 1 if successful and 0 otherwise
 */
int initialize_map (char *image) 
{
	// map
	image_header = (Elf32_Ehdr *) mapimage (image);
	if (image_header == MAP_FAILED)
		return 0;

	section_headers = (Elf32_Shdr *) ( (char *) image_header + image_header->e_shoff );
	sec_strings = (char *) image_header + section_headers[image_header->e_shstrndx].sh_offset;

	// we need symbol table to go
	int section;

	for (symbols_hdr = NULL, section = 1; section < image_header->e_shnum; section++) 
		if (section_headers[section].sh_type == SHT_SYMTAB) {
			symbols_hdr = section_headers + section;
			
			break;
		}
	
	if (! symbols_hdr) {
		LOGERR ("%s: no symbols\n", image);
		return 0;
	}

	// symbols' shortcuts
	syms = (Elf32_Sym *) ((char *) image_header + symbols_hdr->sh_offset);
	strtab = (char *) image_header + section_headers[symbols_hdr->sh_link].sh_offset;

	// are there any relocs 
	for (section = 1; section < image_header->e_shnum; section++) {
		unsigned int info = section_headers[section].sh_info;

		if (info >= image_header->e_shnum) 
			continue;

		if (section_headers[section].sh_type == SHT_REL) 
			break;
	}
	if (section == image_header->e_shnum) {
		LOGWARN ("Nothing to relocate\n");
		return 0;
	}

	return 1;
}

static int symbol_to_idx (char *unknown_symbol) 
{

	// iterate symbols
	int idx, number = symbols_hdr->sh_size / sizeof (Elf32_Sym);

	for (idx = 0; idx < number; idx++) {
		// looking for symbol index
		char *symbol = strtab + syms[idx].st_name;

		DEBUG_DRELOCS ("processing symbol %s\n", symbol);

		if (! strcmp (symbol, unknown_symbol)) {
			// resolved
			DEBUG_DRELOCS ("resolved %s, ndx %d\n", symbol, idx);
			return idx;
		}
	}

	return -1;
}

/* to fill indeces for a number of symbols in a single pass
 * number of symbols to be returned
 *
 * not used currently, each symbol to have a pass
 */
void fill_idx_symbols (struct symbol *unknown_symbols, int unknown_num) 
{
	// iterate symbols
	int idx, number = symbols_hdr->sh_size / sizeof (Elf32_Sym);

	for (idx = 0; idx < number; idx++) {
		// looking for symbol index
		char *symbol = strtab + syms[idx].st_name;
		int u_idx;

		DEBUG_DRELOCS ("processing symbol %s\n", symbol);

		for (u_idx = 0; u_idx < unknown_num; u_idx++)
			if (! strcmp (symbol, unknown_symbols[u_idx].name)) {
				// resolved
				unknown_symbols[u_idx].idx = idx;
				DEBUG_DRELOCS ("resolved %s, ndx %d\n", symbol, idx);
					
				break; // from unknown reloc enum
			}
	}
}

void tweak_areloc (Elf32_Rel *entry, Elf32_Sym *sym, Elf32_Addr val) {
	Elf32_Addr oldval = * (Elf32_Addr *) (entry)->r_offset; 
	int set_new = 0;

	if (oldval == sym->st_value) {
		* (Elf32_Addr *) entry->r_offset = val;
		return;
	}

	if (oldval == sym->st_value + sym->st_size) {
		DEBUG_DRELOCS ("On the +border offset %d for sym %s\n", 
				oldval - sym->st_value, 
				strtab + sym->st_name);
		set_new = 1;
	}
	else if (oldval + 1 == sym->st_value) {
		DEBUG_DRELOCS ("On the -border offset at %x for sym %s\n", 
				entry->r_offset, 
				strtab + sym->st_name);
		set_new = 1;
	}
	else if (oldval < sym->st_value)
	{
		LOGWARN ("Bad reloc at 0x%x: negative offset %d original value 0x%x for sym %s. "
			 "Not setting.\n", entry->r_offset, 
				oldval - sym->st_value, oldval,
				strtab + sym->st_name);
	}	
	else if (oldval > sym->st_value + sym->st_size) 
	{
		int overflow = oldval - sym->st_value - sym->st_size;
		LOGWARN ("Bad reloc at 0x%x: "
			 "beyond range +%d original value 0x%x for sym %s.\n",
			 entry->r_offset, overflow, oldval, strtab + sym->st_value);
		if (overflow <= 3 * sym->st_size || overflow <= 0x1000)
			set_new = 1;
		else 
			LOGWARN ("Not setting\n");
	}	
	else {
		DEBUG_DRELOCS ("Offset %d for sym %s\n", oldval - sym->st_value, 
				strtab + sym->st_name);
		set_new = 1;
	}

	if (set_new)
		* (Elf32_Addr *) entry->r_offset = val + (oldval - sym->st_value);
}

inline void tweak_relocs_in_section (unsigned section, unsigned sym_idx, Elf32_Addr newval)
{
	Elf32_Rel *entry = (Elf32_Rel *) ( (char *) image_header + section_headers[section].sh_offset );
	int entries_num = section_headers[section].sh_size / sizeof (Elf32_Rel);
	size_t pagesize = sysconf(_SC_PAGESIZE);
	size_t pagemask = ~(pagesize - 1);

	int entry_idx;

	DEBUG_DRELOCS ("processing relocation section %s (%d entries)\n", sec_strings + section_headers[section].sh_name, entries_num);
	
	for (entry_idx = 0; entry_idx < entries_num; entry_idx++, entry++) {
		switch (ELF32_R_TYPE(entry->r_info)) {
		case R_386_32:
		// case R_386_PC32:
			if (ELF32_R_SYM(entry->r_info) == sym_idx) {
			// entry->r_offset == reloc_symresolved[fix_idx]->entry) {
				int wa_length = ( (pagemask & (entry->r_offset + 3)) > entry->r_offset ) ? 2 * pagesize : pagesize;

				DEBUG_DRELOCS ("\tfound entry 0x%x\n", entry->r_offset);
				DEBUG_DRELOCS ("\tsanity check symbol %s\n", strtab + syms[ELF32_R_SYM(entry->r_info)].st_name );

				int res = mprotect (
					(void *) (pagemask & entry->r_offset),
					wa_length, 
					PROT_READ | PROT_WRITE | PROT_EXEC
					);
				if (res != 0) {
					LOGSYSERR (Debug, "mprotect");
					DEBUG_DRELOCS ("mprotect: code %d\n", res);
				}
				else
					tweak_areloc (entry, &syms[sym_idx], newval);

				res = mprotect (
					(void *) (pagemask & entry->r_offset),
					wa_length, 
					PROT_READ | PROT_EXEC
					);

				// tweak in the position where relocation is
				DEBUG_DRELOCS ("\ttweaked to 0x%x\n", newval );
			}
			continue;

		// default:
			// wrong relocation
		}
	}
}

// tweak relocations for symbol with given idx to newval
static void tweak_relocs (int sym_idx, Elf32_Addr newval)
{
	int section; 		// multiple relocation sections
	for (section = 1; section < image_header->e_shnum; section++) {
		unsigned int info = section_headers[section].sh_info;

		if (info >= image_header->e_shnum) 
			continue;

		if (section_headers[section].sh_type == SHT_RELA) 
			DEBUG_DRELOCS ("rela section\n");

		if ( (section_headers[section].sh_type == SHT_REL) && 
		     (section_headers[info].sh_flags & SHF_ALLOC) ) 
			tweak_relocs_in_section (section, sym_idx, newval);
	}
}

void relocate_symbol (char *symbol, void *newval) 
{
	// find symbol idx
	int idx = symbol_to_idx (symbol);

	if (idx == -1) return;

	DEBUG_DRELOCS ("got sym %s idx %d\n", symbol, idx);

	tweak_relocs (idx, (Elf32_Addr ) newval);
}


void dump_range (void *addr, int num)
{
	char *ptr = addr;
	
	addr = (char *) addr + num;
	for (ptr = (char *) addr; ptr < (char *) addr; ptr++) 
		putchar (*ptr);
}

void *locate_map_using_proc (char *progname)
{
	char cmd[256]; 
	FILE *p; 
	Elf32_Addr addr; 
	
	// sprintf (cmd, "grep %s /proc/%d/maps skip=%d count=1", getpid(), 0x1000, 0x8048);
	sprintf (cmd, "printf 0x; grep %s /proc/%d/maps | grep '^[0-9a-f-]* rw'", progname, getpid());
	// printf ("cmd: %s\n", cmd);

	p = popen (cmd, "r");
	fscanf (p, "%x", & addr);
	fclose (p);

	// printf ("addr: 0x%x\n", addr);
	return (void *) addr;
}

#ifdef DYNAMICRELOCS_SELFCONTAINED
extern void reloc_call();
extern int reloc_val;

int reloc_val_subst = 4;
main (int argc, char *argv[])
{
	getchar ();
	if (! initialize_map (argv[0]))
		return 1;
	printf ("Original reloc_val: %d, subst: %d\n", reloc_val, reloc_val_subst);
	printf ("changing <reloc_val> sym to 0x%x\n", & reloc_val_subst);
	change_module_symbol ("reloc_val", (Elf32_Addr) &reloc_val_subst);
	printf ("now reloc_val: %d, subst: %d\n", reloc_val, reloc_val_subst);
	reloc_call();
	int s = reloc_val;
	printf ("reloc_val: %d\n", reloc_val);
	printf ("reloc_val_subst: %d\n", reloc_val_subst);

	printf ("& reloc_val: 0x%x, & subst: 0x%x\n", & reloc_val, & reloc_val_subst);
}
#endif // DYNAMICRELOCS_SELFCONTAINED
